/**
 * Copyright (C) 2007-2014 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#ifndef INDII_CLUSTER_PEARSONDISTANCE_HPP
#define INDII_CLUSTER_PEARSONDISTANCE_HPP

namespace indii {
/**
 * Pearson distance.
 *
 * @tparam T1 Value type.
 * @tparam T2 Distance type.
 */
template<class T1 = float, class T2 = float>
class PearsonDistance {
public:
  typedef T1 value_type;
  typedef T2 distance_type;

  /**
   * Preprocess point.
   *
   * @param x Point.
   */
  static void prepare(T1& x);

  /**
   * Calculate distance.
   *
   * @param x1 First point.
   * @param x2 Second point, should have same size as first.
   *
   * @return %Distance between the two points.
   */
  static T2 distance(const T1& x1, const T1& x2);
};
}

template<class T1, class T2>
inline void indii::PearsonDistance<T1, T2>::prepare(T1& x) {
  typedef typename T1::value_type T3;

  T3 mu = boost::numeric::ublas::sum(x) / x.size();
  x -= boost::numeric::ublas::scalar_vector<T3>(x.size(), mu);
  T3 norm = boost::numeric::ublas::norm_2(x);
  if (norm > 0.0) {
    x /= norm;
  } else {
    x.clear();
  }
}

template<class T1, class T2>
inline T2 indii::PearsonDistance<T1, T2>::distance(const T1& x1, const T1& x2) {
  /* pre-condition */
  assert(x1.size() == x2.size());

  T2 d = T2(0.5) * (T2(1.0) - boost::numeric::ublas::inner_prod(x1, x2));

  /* relaxed IEEE compliance, so make sure this is bound as expected */
  //d = std::max((T)0.0, std::min(d, (T)1.0));
  return d;
}

#endif
