///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file Branding.h 
 * \brief Contains the definition of the Core::Branding class and the Core::BrandingManager. 
 */
 
#ifndef __OVITO_BRANDING_H
#define __OVITO_BRANDING_H

#include <core/Core.h>

class QNetworkReply;

namespace Core {

class Rollout;
class PropertiesPanel;

/**
 * \brief Abstract base class for branding plugins that modify the overall 
 *        apparance of the application.
 * 
 * This virtual base class can be subclassed by plugins that want to
 * modify the branding of the host application.
 * 
 * \author Alexander Stukowski
 * \sa BrandingManager
 */
class CORE_DLLEXPORT Branding : public PluginClass
{
public:

	/// \brief The default constructor.
	Branding() : PluginClass(), _aboutRollout(NULL) {}

	/// \brief Returns the internal name of the application.
	/// \return The internal name of the application.
	///
	/// This string is passed to QCoreApplication::setApplicationName().
	virtual QString applicationName();

	/// \brief Returns the display name of the application.
	/// \return The name of the application product.
	///
	/// This string is used as caption of the main window.
	virtual QString productName();

	/// \brief Returns the version of the application as a string.
	/// \return A string that contains the version number.
	virtual QString productVersion();

	/// \brief Returns the name of the organization that wrote this application.
	/// \return The developer of the application.
	virtual QString organizationName();

	/// \brief Returns a string that describes the application.
	/// \return A description string that is displayed in the "about" dialog box. 
	virtual QString productDescription();

	/// \brief Returns a string that contains the copyright information
	/// \return A copyright statement that is displayed in the "about" dialog box.
	virtual QString copyrightString();

	/// \brief Returns the URL of the product's website.
	/// \return The URL of the website. 
	virtual QString productWebSite();

	/// \brief Lets the branding display the about panel in the command panel of the main window.
	///        This is called by the system when no scene node is selected.
	/// \param container The properties panel in which to display the about panel.
	virtual void showAboutPanel(PropertiesPanel* container);

	/// \brief Lets the branding hide its about panel.
	/// \param container The properties panel in which the about panel is shown.
	virtual void hideAboutPanel(PropertiesPanel* container);

	/// \brief Returns the default file extension to use for scene files.
	/// \return The default file extension string without the dot. 
	virtual QString sceneFileExtension();

	/// \brief Returns the string that is used as description in the filter box
	///        of the File open and File save dialog boxes.
	/// \return The human-readable description of the file format.
	virtual QString sceneFileFilter();

	/// \brief Is called by the system before the main window is being created.
	virtual void preConfigureMainWindow() {}

	/// \brief Is called by the system after the main window has been created.
	virtual void postConfigureMainWindow() {}

private Q_SLOTS:

	/// Is called by the system when fetching the news webpage from the webserver is finished.
	void onWebRequestFinished(QNetworkReply* reply);

private:

	/// The panel displaying information about the application when no object is selected.
	Rollout* _aboutRollout;
	/// Shows the about webpage.
	QTextBrowser* _aboutLabel;

	Q_OBJECT	
	DECLARE_PLUGIN_CLASS(Branding)
};

/**
 * \brief Manages the registered Branding classes.
 * 
 * \author Alexander Stukowski
 * \sa Branding
 */
class CORE_DLLEXPORT BrandingManager : public QObject
{
	Q_OBJECT

public:

	/// Returns all installed brandings.
	static const QVector<Branding::SmartPtr>& brandings();

	/// Returns the primary branding object that controls the application's title etc.
	static Branding::SmartPtr primaryBranding() {
		OVITO_ASSERT(brandings().empty() == false);
		return brandings().back();
	}

private:

	/// Contains the installed brandings.
	static QVector<Branding::SmartPtr> _brandings;

	/// Releases all installed brandings.
	static void shutdown();

	friend class ApplicationManager;
};

};

#endif // __OVITO_BRANDING_H
