///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/BooleanGroupBoxPropertyUI.h>
#include <core/undo/UndoManager.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/AnimManager.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(BooleanGroupBoxPropertyUI, ReferenceParameterUI)

/******************************************************************************
* Constructor for a Qt property.
******************************************************************************/
BooleanGroupBoxPropertyUI::BooleanGroupBoxPropertyUI(PropertiesEditor* parentEditor, const char* propertyName, const QString& label) :
	PropertyParameterUI(parentEditor, propertyName)
{
	// Create UI widget.
	_groupBox = new QGroupBox(label);
	_groupBox->setCheckable(true);
	connect(_groupBox, SIGNAL(clicked(bool)), this, SLOT(updatePropertyValue()));
}

/******************************************************************************
* Constructor for a PropertyField property.
******************************************************************************/
BooleanGroupBoxPropertyUI::BooleanGroupBoxPropertyUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& propField) :
	PropertyParameterUI(parentEditor, propField)
{
	// Create UI widget.
	_groupBox = new QGroupBox(propField.displayName());
	_groupBox->setCheckable(true);
	connect(_groupBox, SIGNAL(clicked(bool)), this, SLOT(updatePropertyValue()));
}

/******************************************************************************
* Destructor, that releases all GUI controls.
******************************************************************************/
BooleanGroupBoxPropertyUI::~BooleanGroupBoxPropertyUI()
{
	// Release GUI controls. 
	delete groupBox(); 
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to.
******************************************************************************/
void BooleanGroupBoxPropertyUI::resetUI()
{
	PropertyParameterUI::resetUI();
	
	if(groupBox()) 
		groupBox()->setEnabled(editObject() != NULL && isEnabled());
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to.
******************************************************************************/
void BooleanGroupBoxPropertyUI::updateUI()
{
	PropertyParameterUI::updateUI();

	if(groupBox() && editObject()) {
		QVariant val(false);
		if(propertyName()) {
			val = editObject()->property(propertyName());
			OVITO_ASSERT_MSG(val.isValid(), "BooleanGroupBoxPropertyUI::updateUI()", QString("The object class %1 does not define a property with the name %2 that can be cast to bool type.").arg(editObject()->metaObject()->className(), QString(propertyName())).toLocal8Bit().constData());
			if(!val.isValid()) {
				throw Exception(tr("The object class %1 does not define a property with the name %2 that can be cast to bool type.").arg(editObject()->metaObject()->className(), QString(propertyName())));
			}
		}
		else if(propertyField()) {
			val = editObject()->getPropertyFieldValue(*propertyField());
			OVITO_ASSERT(val.isValid());
		}
		groupBox()->setChecked(val.toBool());
	}
}

/******************************************************************************
* Sets the enabled state of the UI.
******************************************************************************/
void BooleanGroupBoxPropertyUI::setEnabled(bool enabled)
{
	if(enabled == isEnabled()) return;
	PropertyParameterUI::setEnabled(enabled);
	if(groupBox()) groupBox()->setEnabled(editObject() != NULL && isEnabled());
}

/******************************************************************************
* Takes the value entered by the user and stores it in the property field
* this property UI is bound to.
******************************************************************************/
void BooleanGroupBoxPropertyUI::updatePropertyValue()
{
	if(groupBox() && editObject()) {

		UNDO_MANAGER.beginCompoundOperation(tr("Change parameter"));

		if(propertyName()) {
			if(!editObject()->setProperty(propertyName(), groupBox()->isChecked())) {
				OVITO_ASSERT_MSG(false, "BooleanGroupBoxPropertyUI::updatePropertyValue()", QString("The value of property %1 of object class %2 could not be set.").arg(QString(propertyName()), editObject()->metaObject()->className()).toLocal8Bit().constData());
			}
		}
		else if(propertyField()) {
			editObject()->setPropertyFieldValue(*propertyField(), groupBox()->isChecked());
		}

		UNDO_MANAGER.endCompoundOperation();
	}
}

};

