/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Qt includes
#include <QtDebug>


/////////////////////// Local includes
#include "PropListHolder.hpp"


namespace MsXpS
{

namespace libXpertMass
{


/*!
\class MsXpS::libXpertMass::PropListHolder
\inmodule libXpertMass
\ingroup ThePropSystem
\inheaderfile PropListHolder.hpp

\brief The PropListHolder class is the base class for a number of
classes that need storing Prop instances.

\sa Modif, Monomer, Oligomer, PkaPhPi,
*/


/*!
\variable int MsXpS::libXpertMass::PropListHolder::m_propList

\brief The m_propList stores pointer to \l Prop instances.
*/

/*!
\brief Constructs a PropListHolder instance.
*/
PropListHolder::PropListHolder()
{
}

/*!
\brief Constructs a PropListHolder instance as a copy of \a other.

The Prop instances in the \a other PropListHolder instance's member list of
Prop instances are duplicated and stored in this instance using their own copy
constructor.
*/
PropListHolder::PropListHolder(const PropListHolder &other)
{
  for(int iter = 0; iter < other.m_propList.size(); ++iter)
    {
      Prop *prop = other.m_propList.at(iter);
      Q_ASSERT(prop);
      //qDebug() << "Prop name:" << prop->name();

      // Each Prop-derived class produces a derived Prop class.
      m_propList.append(prop->cloneOut());
    }
}

/*!
\brief Destructs this PropListHolder instance.
*/
PropListHolder::~PropListHolder()
{
  while(!m_propList.isEmpty())
    delete(m_propList.takeFirst());
}

/*!
\brief Assigns \a other to this PropListHolder instance.

The Prop instances in the \a other PropListHolder instance's member list of
Prop instances are duplicated and stored in this instance using their own copy
constructor.

Returns a reference to this PropListHolder instance.
*/
PropListHolder &
PropListHolder::operator=(const PropListHolder &other)
{
  if(&other == this)
    return *this;

  for(int iter = 0; iter < other.m_propList.size(); ++iter)
    {
      Prop *prop = other.m_propList.at(iter);
      Q_ASSERT(prop);

      m_propList.append(prop->cloneOut());
    }

  return *this;
}

/*!
\brief Returns a const reference to the member list of Prop instances.
*/
const QList<Prop *> &
PropListHolder::propList() const
{
  return m_propList;
}

/*!
\brief Returns a non-const reference to the member list of Prop instances.
*/
QList<Prop *> &
PropListHolder::propList()
{
  return m_propList;
}

/*!
\brief Searches in the member list of Prop instances a Prop having \a name.

If the Prop instance was found and \a index is non-nullptr, its index in the
list is set to this parameter.

Returns the found Prop instance.
*/
Prop *
PropListHolder::prop(const QString &name, int *index)
{
  for(int iter = 0; iter < m_propList.size(); ++iter)
    {
      Prop *localProp = m_propList.at(iter);
      Q_ASSERT(localProp);

      if(localProp->name() == name)
        {
          if(index)
            *index = iter;

          return localProp;
        }
    }

  return 0;
}


/*!
\brief Searches in the member list of Prop instances a Prop having \a name.

If the Prop instance was found and \a prop is non-nullptr, its pointer is set
to this parameter.

Returns the index of the found Prop instance.
*/
int
PropListHolder::propIndex(const QString &name, Prop *prop)
{
  for(int iter = 0; iter < m_propList.size(); ++iter)
    {
      Prop *localProp = m_propList.at(iter);
      Q_ASSERT(localProp);

      if(localProp->name() == name)
        {
          if(prop)
            prop = localProp;

          return iter;
        }
    }

  return -1;
}

/*!
\brief Adds \a prop to the member list of Prop instances.

Returns true.
*/
bool
PropListHolder::appendProp(Prop *prop)
{
  m_propList.append(prop);

  return true;
}


/*!
\brief Removes \a prop from the member list of Prop instances.

Returns true if \a prop was removed, false otherwise.
*/
bool
PropListHolder::removeProp(Prop *prop)
{
  if(m_propList.removeAll(prop))
    return true;

  return false;
}


/*!
\brief Removes the Prop instance having \a name from the member list of Prop
instances.

If more than one Prop instance by the \a name are in the list, only the first
encountered Prop instance is removed.

Returns true if a Prop instance was removed, false otherwise.
*/
bool
PropListHolder::removeProp(const QString &name)
{
  for(int iter = 0; iter < m_propList.size(); ++iter)
    {
      if(m_propList.at(iter)->name() == name)
        {
          m_propList.removeAt(iter);

          return true;
        }
    }

  return false;
}

} // namespace libXpertMass

} // namespace MsXpS
