/* devices.c
 *
 * (c) 2015 Markus Heinz
 *
 * This software is licensed under the terms of the GPL.
 * For details see file COPYING.
 */

#include "config.h"

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>

#if (HOST_OS == SOLARIS)
#include <sys/prnio.h>
#endif

#include "inklevel.h"
#include "internal.h"
#include "devices.h"
#include "bjnp.h"

#if (HOST_OS != SOLARIS)
#include "libusb-utils.h"
#undef HAVE_IEEE1284_H
#include <ieee1284.h>
#else

int perform_device_id_ioctl(int port, const char *device_file, 
                            const int portnumber, char *device_id);


int perform_device_id_ioctl(const int port, const char *device_file, 
                            const int portnumber, char *device_id) {
  int fd;
  char my_device_file[256];
  struct prn_1284_device_id id;
  int rc;
  int size;
  char tmp[BUFLEN];

  sprintf(my_device_file, device_file, portnumber);
  fd = open(my_device_file, O_RDONLY);

  if (fd == -1) {
    switch (port) {
    case USB:
      return DEV_USB_LP_INACCESSIBLE;
    case PARPORT:
      return DEV_LP_INACCESSIBLE;
    }
  }

  /* get the IEEE 1284 device id */
  memset(&id, 0, sizeof (id));
  memset(&tmp, 0, sizeof (tmp));
  id.id_len = sizeof (tmp);
  id.id_data = tmp;
  
  rc = ioctl(fd, PRNIOC_GET_1284_DEVID, &id);
  
  if (rc < 0) {
    close(fd);
    return COULD_NOT_GET_DEVICE_ID;
  }

  close(fd);
    
  size = id.id_rlen;
  tmp[size] = '\0';
  
  if (size > 0) {
    strncpy(device_id, tmp, size + 1);
    return OK;
  } else {
    return COULD_NOT_GET_DEVICE_ID;
  }
}

#endif

int get_device_id(const int port, const char *device_file, 
                  const int portnumber, char *device_id) {

#if (HOST_OS != SOLARIS)
  struct parport_list parports;
#endif

  char tmp[BUFLEN];
  char device_file1[256];
  int size;
  int fd;
  int result = COULD_NOT_GET_DEVICE_ID;

  if (port == PARPORT ) {

#if (HOST_OS != SOLARIS)

    /* check if we have appropriate permissions */

#if (HOST_OS == LINUX)

    sprintf(device_file1, "/dev/parport%d", portnumber);

    if ((fd = open(device_file1, O_RDWR)) < 0) {
      return DEV_PARPORT_INACCESSIBLE;
    }

    close(fd);

    sprintf(device_file1, "/dev/lp%d", portnumber);
    
    if ((fd = open(device_file1, O_RDWR)) < 0) {
      return DEV_LP_INACCESSIBLE;
    }

    close(fd);

#elif (HOST_OS == FREEBSD)

    sprintf(device_file1, "/dev/lpt%d", portnumber);
    
    if ((fd = open(device_file1, O_RDWR)) < 0) {
      return DEV_LP_INACCESSIBLE;
    }

    close(fd);

#endif

    /* for linux and freebsd use libieee1284 to get the device id from 
       parallel port */

    if (ieee1284_find_ports(&parports, 0) == E1284_OK) {
      if (portnumber < parports.portc) {
	size = ieee1284_get_deviceid(parports.portv[portnumber], -1, 
                                     F1284_FRESH, tmp, BUFLEN);
	if (size >= 2) {
	  strncpy(device_id, tmp + 2, size - 2);
	  return OK;
	}
      }
    }

    return COULD_NOT_GET_DEVICE_ID;

#else
    
    /* solaris */
    return perform_device_id_ioctl(port, "/dev/ecpp%d", portnumber, device_id);

#endif
    
  } else if (port == USB) {

#if (HOST_OS != SOLARIS)

    libusb_context *ctx = NULL;
    usb_printer *printer = NULL;

    if (init_usb(ctx) != USB_SUCCESS) {
      return COULD_NOT_GET_DEVICE_ID;
    }

    printer = find_printer(ctx, portnumber);

    if (printer != NULL) {
      result = open_device_handle(printer);

      if (result == USB_SUCCESS) {
	result = get_usb_device_id(printer, device_id, BUFLEN);
        release_device_handle(printer);
      }
      
      free(printer);
    }

    shutdown_usb(ctx);

    if (result == USB_SUCCESS)
      return OK;
    else
      return COULD_NOT_GET_DEVICE_ID;

#else

    /* solaris */
    return perform_device_id_ioctl(port, "/dev/usb/printer%d", portnumber, 
                                   device_id);

#endif

  } else if (port == CUSTOM_BJNP)  {
    return bjnp_get_id_from_named_printer(portnumber, device_file, device_id);
  } else if (port == BJNP) {
    return bjnp_get_id_from_printer_port(portnumber, device_id);
  } else {
    return UNKNOWN_PORT_SPECIFIED;
  }
}

int open_printer_device(const int port, const char *custom_device_file,
                        const int portnumber) {
  char device_file[256];
  int fd;

#if (HOST_OS == LINUX)

  if (port == PARPORT) {
    sprintf(device_file, "/dev/lp%d", portnumber);
  } else {
    return UNKNOWN_PORT_SPECIFIED;
  }

#elif (HOST_OS == FREEBSD) 

  if (port == PARPORT) {
    sprintf(device_file, "/dev/lpt%d", portnumber);
  } else {
    return UNKNOWN_PORT_SPECIFIED;
  }

#elif (HOST_OS == SOLARIS)

  if (port == USB) {
    sprintf(device_file, "/dev/usb/printer%d", portnumber);
  } else if (port == PARPORT) {
    sprintf(device_file, "/dev/ecpp%d", portnumber);
  } else {
    return UNKNOWN_PORT_SPECIFIED;
  }

#endif

#ifdef DEBUG
  printf("Device file: %s\n", device_file);
#endif

  fd = open(device_file, O_RDWR);

  if (fd == -1) {

#ifdef DEBUG
    printf("Could not open %s\n", device_file);
#endif
  }
  
  if (fd == -1) {
    if (port == USB) {
      return DEV_USB_LP_INACCESSIBLE;
    } else {
      return DEV_LP_INACCESSIBLE;
    }
  } else {
    return fd;
  }
}
