#!/usr/bin/env python

"""
A Web interface to the user profile.

Copyright (C) 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imipweb.resource import FormUtilities, ResourceClient

# Fake gettext method for strings to be translated later.

_ = lambda s: s

class ProfilePage(ResourceClient, FormUtilities):

    "A request handler for the user profile page."

    # See: imiptools.config, imiptools.profile
    # NOTE: This could be defined in another format and generated or computed
    # NOTE: for this class and for the documentation.
    # See: docs/tools/profile_prefs.sh

    pref_labels = [
        ("participating"         , _("Participate in the calendar system")),
        ("CN"                    , _("Your common name")),
        ("LANG"                  , _("Language")),
        ("TZID"                  , _("Time zone/regime")),
        ("incoming"              , _("How to present incoming calendar messages")),
        ("freebusy_sharing"      , _("Share free/busy information")),
        ("freebusy_bundling"     , _("Bundle free/busy details with messages")),
        ("freebusy_publishing"   , _("Publish free/busy details via the Web")),
        ("freebusy_messages"     , _("Deliver details of received free/busy messages")),
        ("add_method_response"   , _("How to respond to messages adding events")),
        ("event_refreshing"      , _("How to handle event refresh requests")),
        ("organiser_replacement" , _("Recognise whom as a new organiser of an event?")),
        ]

    def handle_request(self):
        args = self.env.get_args()
        save = args.has_key("save")
        cancel = args.has_key("cancel")
        action = save or cancel

        if not action:
            return ["action"]

        # Check the validation token.

        if not self.check_validation_token():
            return ["token"]

        if save:
            errors = self.update_preferences()
            if errors:
                return errors
            else:
                self.redirect(self.link_to())

        elif cancel:
            self.redirect(self.link_to())

        return None

    def update_preferences(self):

        "Update the stored preferences."

        settings = self.get_current_preferences()
        prefs = self.get_preferences()
        errors = []

        for name, value in settings.items():
            choices = prefs.known_key_choices.get(name)
            if choices and not choices.has_key(value):
                errors.append(name)

        if errors:
            return errors

        for name, value in settings.items():
            prefs[name] = value

    # Request logic methods.

    def is_initial_load(self):

        "Return whether the event is being loaded and shown for the first time."

        return not self.env.get_args().has_key("editing")

    def get_stored_preferences(self):

        "Return stored preference information for the current user."

        prefs = self.get_preferences()
        return dict(prefs.items())

    def get_current_preferences(self):

        "Return the preferences currently being edited."

        if self.is_initial_load():
            return self.get_stored_preferences()
        else:
            return dict([(name, values and values[0] or "") for (name, values) in self.env.get_args().items()])

    # Output fragment methods.

    def show_preferences(self, errors=None):

        "Show the preferences, indicating any 'errors' in the output."

        _ = self.get_translator()

        page = self.page
        settings = self.get_current_preferences()
        prefs = self.get_preferences()

        # Add a hidden control to help determine whether editing has already begun.

        self.control("editing", "hidden", "true")

        # Show the range of preferences, getting all possible entries and using
        # configuration defaults.

        page.table(class_="profile", cellspacing=5, cellpadding=5)
        page.thead()
        page.tr()
        page.th(_("Preferences"), class_="mainheading", colspan=2)
        page.tr.close()
        page.thead.close()
        page.tbody()

        for name, label in self.pref_labels:
            value = settings.get(name)
            default = prefs.known_keys.get(name)
            choices = prefs.known_key_choices.get(name)

            page.tr()
            page.th(class_="profileheading %s%s" % (name, errors and name in errors and " error" or ""))
            page.label(_(label), for_=name)
            page.th.close()
            page.td()

            # For unrestricted fields, show a text field.

            if not choices:
                page.input(name=name, value=(value or default), type="text", class_="preference", id_=name)

            # Otherwise, obtain the choices, localise the labels and show a
            # menu control.

            else:
                choices = [(key, _(value_label)) for (key, value_label) in choices.items()]
                choices.sort()
                self.menu(name, default, choices, value is not None and [value] or None, class_="preference")

            page.td.close()
            page.tr.close()

        page.tbody.close()
        page.table.close()

    def show_controls(self):

        "Show controls for performing actions."

        _ = self.get_translator()

        page = self.page

        page.p(class_="controls")
        page.input(name="save", type="submit", value=_("Save"))
        page.input(name="cancel", type="submit", value=_("Cancel"))
        page.p.close()

    # Full page output methods.

    def show(self):

        "Show the preferences of a user."

        page = self.page
        errors = self.handle_request()

        if not errors:
            return True

        _ = self.get_translator()

        self.new_page(title=_("Profile"))
        page.form(method="POST")
        self.validator()
        self.show_preferences(errors)
        self.show_controls()
        page.form.close()

        return True

# vim: tabstop=4 expandtab shiftwidth=4
